
/* Copyright (C) 2002-2018 RealVNC Ltd. All Rights Reserved. */

#ifndef UUID_269fdfdc_51e3_483e_80cb_573aa4c31d7d
#define UUID_269fdfdc_51e3_483e_80cb_573aa4c31d7d

/**
 * \file vncsbp.h
 */


#include "vnccall.h"
#include "vncint.h"
#include "vncsbptypes.h"
#include "vncsbpserialize.h"
#include "vnccdb.h"

#ifdef __cplusplus
extern "C"
{
#endif


/** \defgroup sbp-sink SBP sink functions
 * @{
 */

/** \brief Obtain the state of an object on the source.
 *
 * Sends a Get packet to the remote endpoint to request the state of an
 * object. The remote object is specified by UID - use VNCSBPHash() to convert
 * an object name into a UID. Any response will be passed to the application
 * via VNCSBPSinkGetResponseCallback(). If authentication is requested,
 * the appropriate authentication challenge callback may be called prior to
 * the response callback.
 *
 * Applications may pass arbitary context via the `pCommandContext` parameter.
 * This parameter will be regarded as opaque by the SDK and will be passed into
 * all callback functions applicable to the same command sequence, so may be
 * used to tie together requests, responses and authentication challenges.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID of remote service.
 * \param[in]   pCommandContext Application context for this command sequence.
 * \param[in]   objectUID       UID of object to query.
 *
 * \return      ::VNCCDBErrorNone indicates success, any other value failure.
 */
typedef VNCCDBError VNCCALL
VNCSBPSinkSendGetRequest(VNCCDBEndpoint *pEndpoint,
                         VNCCDBServiceId id,
                         void *pCommandContext,
                         VNCSBPUID objectUID);


/** \brief Change the state of an object on the source.
 *
 * Sends a Set to the remote endpoint to update the specified object
 * with the payload data.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID of remote service.
 * \param[in]   pCommandContext Application context for this command sequence.
 * \param[in]   objectUID       UID of object to update.
 * \param[in]   payload         Pointer to serialized fields to update.
 * \param[in]   payloadSize     Bytes available in `payload`.
 *
 * \return      ::VNCCDBErrorNone indicates success, any other value failure.
 */
typedef VNCCDBError VNCCALL
VNCSBPSinkSendSetRequest(VNCCDBEndpoint *pEndpoint,
                         VNCCDBServiceId id,
                         void *pCommandContext,
                         VNCSBPUID objectUID,
                         const vnc_uint8_t *payload,
                         size_t payloadSize);


/** \brief Subscribe to updates for a specified object.
 *
 * Sends a Subscribe to the remote endpoint to request periodic updates
 * to the state of a specified object. The `trigger` parameter can specify
 * either updates at fixed intervals with ::VNCSBPSubscriptionTypeRegularInterval,
 * in which case `intervalMS` specifies the interval time in milliseconds,
 * or updates only when the value of the object changes with
 * ::VNCSBPSubscriptionTypeOnChange, in which case `intervalMS` is ignored.
 *
 * The `trigger` parameter may also be ::VNCSBPSubscriptionTypeAutomatic which
 * indicates that the remote endpoint may select whichever of the two options
 * is most appropriate for the object - in this case the `intervalMS` parameter
 * must always be specified appropriately, in the event that the remote endpoint
 * elects to send regular updates.
 *
 * Any responses sent back as a result of the subscription will be passed to
 * the application via VNCSBPSinkGetResponseCallback(). If it is necessary 
 * for the application to differentiate between responses to concurrent Get
 * and Subscribe requests, the `pCommandContext` parameter can be used.
 *
 * The `intervalMS` parameter must be no greater than
 * ::VNCSBPMaximumSubscriptionInterval - if the value provided is too large,
 * ::VNCCDBErrorInvalidParameter will be returned.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID of remote service.
 * \param[in]   pCommandContext Application context for this command sequence.
 * \param[in]   objectUID       UID of object to subscribe.
 * \param[in]   trigger         Type of subscription to request.
 * \param[in]   intervalMS      Update interval in milliseconds.
 *
 * \return      ::VNCCDBErrorNone indicates success, any other value failure.
 */
typedef VNCCDBError VNCCALL
VNCSBPSinkSendSubscribeRequest(VNCCDBEndpoint *pEndpoint,
                               VNCCDBServiceId id,
                               void *pCommandContext,
                               VNCSBPUID objectUID,
                               VNCSBPSubscriptionType trigger,
                               vnc_uint32_t intervalMS);


/** \brief Cancel a previously-sent command.
 *
 * Sends a Cancel for a previously-sent Get, Set or
 * Subscribe. The command type and object that were used in the
 * original request must be specified. If no such request is outstanding,
 * VNCCDBErrorNotFound will be returned and no Cancel will be sent.
 *
 * Due to the asynchronous nature of the SDK applications cannot assume that
 * a response won't be received until the response to the Cancel message
 * is received and passed via VNCSBPSinkCancelResponseCallback().
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID of remote service.
 * \param[in]   pCommandContext Application per-request context.
 * \param[in]   commandType     Request type to cancel.
 * \param[in]   objectUID       Object UID from original request.
 *
 * \return      ::VNCCDBErrorNone indicates success, any other value failure.
 */
typedef VNCCDBError VNCCALL
VNCSBPSinkSendCancelRequest(VNCCDBEndpoint *pEndpoint,
                            VNCCDBServiceId id,
                            void *pCommandContext,
                            VNCSBPCommandType commandType,
                            VNCSBPUID objectUID);


/** \brief Callback from the SDK when a response with object data is received.
 *
 * This callback is called in response to either sending a Get or an
 * oustanding Subscribe request. Appropriate use of the `pCommandContext`
 * parameter can be used to differentiate between these two cases if necssary.
 *
 * This callback is only called if a successful response is received, otherwise
 * VNCSBPSinkRequestErrorCallback() is called instead.
 *
 * \note For responses received in response to a Subscribe request, note
 *       that the subscription is terminated on any error and must be manually
 *       re-initiated by the application if the error proves non-fatal.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID of remote service.
 * \param[in]   pSinkContext    Application per-service context.
 * \param[in]   pCommandContext Application per-request context.
 * \param[in]   commandType     Request command which prompted this response.
 * \param[in]   objectUID       Object UID that was requested.
 * \param[in]   payload         Pointer to serialized payload from response.
 * \param[in]   payloadSize     Bytes available in `payload`.
 *
 * \return Return ::VNCCDBErrorNone if the response was handled successfully.
 *         Other error values will cause the service to be terminated. For
 *         example, if a call to deserialize the payload fails, it will
 *         usually be appropriate to return the deserialization error here.
 */
typedef VNCCDBError VNCCALL
VNCSBPSinkGetResponseCallback(VNCCDBEndpoint *pEndpoint,
                              VNCCDBServiceId id,
                              void *pSinkContext,
                              void *pCommandContext,
                              VNCSBPCommandType commandType,
                              VNCSBPUID objectUID,
                              const vnc_uint8_t *payload,
                              size_t payloadSize);


/** \brief Callback from the SDK when a response to a Set is received.
 *
 * This callback is only called if a successful response is received, otherwise
 * VNCSBPSinkRequestErrorCallback() is called instead.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID of remote service.
 * \param[in]   pSinkContext    Application per-service context.
 * \param[in]   pCommandContext Application context for this command sequence.
 * \param[in]   objectUID       Object UID that was requested.
 *
 * \return Return ::VNCCDBErrorNone if the response was handled successfully.
 *         Other error values will cause the service to be terminated.
 */
typedef VNCCDBError VNCCALL
VNCSBPSinkSetResponseCallback(VNCCDBEndpoint *pEndpoint,
                              VNCCDBServiceId id,
                              void *pSinkContext,
                              void *pCommandContext,
                              VNCSBPUID objectUID);


/** \brief Callback from the SDK on an initial response to a Subscribe.
 *
 * The initial response to a Subscribe message does not contain any object
 * updates, it simply indicates whether or not the subscription was accepted.
 * This callback is invoked on a successful response. In the case of an
 * unsuccessful response, VNCSBPSinkRequestErrorCallback() is called instead.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID of remote service.
 * \param[in]   pSinkContext    Application per-service context.
 * \param[in]   pCommandContext Application context for this command sequence.
 * \param[in]   objectUID       Object UID that was requested.
 *
 * \return Return ::VNCCDBErrorNone if the response was handled successfully.
 *         Other error values will cause the service to be terminated.
 */
typedef VNCCDBError VNCCALL
VNCSBPSinkSubscribeResponseCallback(VNCCDBEndpoint *pEndpoint,
                                    VNCCDBServiceId id,
                                    void *pSinkContext,
                                    void *pCommandContext,
                                    VNCSBPUID objectUID);


/** \brief Callback from the SDK when a response to a Cancel is received.
 *
 * This callback is only called if a successful response is received, otherwise
 * VNCSBPSinkRequestErrorCallback() is called instead.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID of remote service.
 * \param[in]   pSinkContext    Application per-service context.
 * \param[in]   pCommandContext Application context for this command sequence.
 * \param[in]   commandType     Request type that was cancelled.
 * \param[in]   objectUID       Object UID for the cancelled request.
 *
 * \return Return ::VNCCDBErrorNone if the response was handled successfully.
 *         Other error values will cause the service to be terminated.
 */
typedef VNCCDBError VNCCALL
VNCSBPSinkCancelResponseCallback(VNCCDBEndpoint *pEndpoint,
                                 VNCCDBServiceId id,
                                 void *pSinkContext,
                                 void *pCommandContext,
                                 VNCSBPCommandType commandType,
                                 VNCSBPUID objectUID);


/** \brief Callback from the SDK when a request fails or times out.
 *
 * This callback is invoked when the remote service doesn't respond within the
 * time mandated by the SBP standard, or responds with an error, or in cases
 * where there is some other issue sending the request (e.g. network problems).
 *
 * The objectUID parameter is only valid if commandType is one of
 * ::VNCSBPCommandTypeGet, ::VNCSBPCommandTypeSet and ::VNCSBPCommandTypeSubscribe.
 * The sbpError parameter is only valid if sdkError is either
 * ::VNCCDBErrorLocalSBPError (in the case of an SBP error code generated within
 * the SDK) or ::VNCCDBErrorRemoteSBPError (in the case of an error response from
 * the remote service).
 *
 * After any error, the application can consider the original request cancelled
 * and is free to repeat it.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID of remote service.
 * \param[in]   pSinkContext    Application per-service context.
 * \param[in]   pCommandContext Application per-request context.
 * \param[in]   commandType     Request type which has errored.
 * \param[in]   objectUID       UID of remote object for which request was sent.
 * \param[in]   sdkError        SDK error code.
 * \param[in]   sbpError        SBP protocol error code.
 *
 * \return Return ::VNCCDBErrorNone if the error was handled successfully.
 *         Other error values will cause the service to be terminated.
 */
typedef VNCCDBError VNCCALL
VNCSBPSinkRequestErrorCallback(VNCCDBEndpoint *pEndpoint,
                               VNCCDBServiceId id,
                               void *pSinkContext,
                               void *pCommandContext,
                               VNCSBPCommandType commandType,
                               VNCSBPUID objectUID,
                               VNCCDBError sdkError,
                               VNCSBPProtocolError sbpError);


/** \brief Callback from the SDK when an AuthenticationChallenge is received
 *         specifying service-specific authentication.
 *
 * Applications should always call VNCSBPSinkSendServiceAuthResponse() in 
 * response to this callback, either to send a valid authentication 
 * response if defined for the service, or to send back 
 * ::VNCSBPProtocolErrorFeatureNotSupported.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID of remote service.
 * \param[in]   pSinkContext    Application per-service context.
 * \param[in]   pCommandContext Application context for this command sequence.
 * \param[in]   commandType     Request type which provoked this challenge.
 * \param[in]   objectUID       UID of remote object for which request was sent.
 * \param[in]   payload         Pointer to serialized payload from challenge..
 * \param[in]   payloadSize     Bytes available in `payload`.
 *
 * \return Return ::VNCCDBErrorNone if the challenge was handled successfully.
 *         Other error values will cause the service to be terminated.
 */
typedef VNCCDBError VNCCALL
VNCSBPSinkServiceAuthChallengeCallback(VNCCDBEndpoint *pEndpoint,
                                       VNCCDBServiceId id,
                                       void *pSinkContext,
                                       void *pCommandContext,
                                       VNCSBPCommandType commandType,
                                       VNCSBPUID objectUID,
                                       const vnc_uint8_t *payload,
                                       size_t payloadSize);


/** \brief Send an AuthenticationResponse for a service-specific challenge.
 *
 * An AuthenticationResponse should only be sent once the SDK has indicated
 * receipt of service-specific challenge by calling
 * VNCSBPSinkServiceAuthChallengeCallback(). The application should either
 * pass ::VNCSBPProtocolErrorNone as the 'errorCode' parameter and provide a
 * pointer to a serialized payload as the 'payload' parameter, or pass
 * another error as the 'errorCode' parameter (typically
 * ::VNCSBPProtocolErrorFeatureNotSupported) and NULL for the 'payload'
 * parameter.
 *
 * The 'commandType' and 'objectUID' parameters must be the parameters passed
 * to VNCSBPSinkServiceAuthChallengeCallback() function.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID of remote service.
 * \param[in]   commandType     Request type which provoked this challenge.
 * \param[in]   objectUID       UID of remote object for which request was sent.
 * \param[in]   errorCode       Error code to include in AuthenticationResponse.
 * \param[in]   payload         Pointer to serialized payload to include.
 * \param[in]   payloadSize     Number of valid bytes in 'payload'.
 *
 * \return      ::VNCCDBErrorNone indicates success, any other value failure.
 */
typedef VNCCDBError VNCCALL
VNCSBPSinkSendServiceAuthResponse(VNCCDBEndpoint *pEndpoint,
                                  VNCCDBServiceId id,
                                  void *pCommandContext,
                                  VNCSBPCommandType commandType,
                                  VNCSBPUID objectUID,
                                  VNCSBPProtocolError errorCode,
                                  const vnc_uint8_t *payload,
                                  size_t payloadSize);


/** \brief Check whether the SBP Source is responding.
 *
 * Sends an AliveRequest packet to the SBP Source which should respond
 * within 5 seconds. Upon receiving a response VNCSBPSinkServiceAliveCallback()
 * is invoked, or if the 5 second timeout expires with no response then 
 * VNCSBPSinkRequestErrorCallback() will be invoked instead.
 *
 * To send a single AliveRequest, call with repeatIntervalSecs set to 0. The
 * SDK can automatically repeat the liveness checks at a specified interval.
 * To do this, set repeatIntervalSecs to a positive number of seconds between
 * checks. To disable the automatic repeat, set repeatIntervalSecs to
 * a negative number.
 *
 * Any outstanding repeating alive check will always be cancelled on a
 * timeout response, to avoid repeatedly calls to the application callback.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID of remote service.
 * \param[in]   pCommandContext Application per-request context.
 * \param[in]   repeatIntervalSecs
 *                              0 will send a one-off request.
 *                              +ve value will repeat at the specified interval.
 *                              -ve value will cancel any currently configured repeat.
 *
 * \return      ::VNCCDBErrorNone indicates success, any other value failure.
 */
typedef VNCCDBError VNCCALL
VNCSBPSinkSendServiceAliveRequest(VNCCDBEndpoint *pEndpoint,
                                  VNCCDBServiceId id,
                                  void *pCommandContext,
                                  int repeatIntervalSecs);


/** \brief Called by the SDK if the endpoint passes a liveness check.
 *
 * If the application has called VNCSBPSinkSendServiceAliveRequest() then the
 * SBP Source should respond within 5 seconds and, upon receiving a successful
 * response, the SDK call invoke this callback.
 *
 * If no response is received within 5 seconds, 
 * VNCSBPSinkRequestErrorCallback() is invoked instead.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID of remote service.
 * \param[in]   pSinkContext    Application per-service context.
 * \param[in]   pCommandContext Application per-request context.
 *
 * \return Return ::VNCCDBErrorNone if the callback was handled successfully.
 *         Other error values will cause the service to be terminated.
 */
typedef VNCCDBError VNCCALL
VNCSBPSinkServiceAliveCallback(VNCCDBEndpoint *pEndpoint,
                               VNCCDBServiceId id,
                               void *pSinkContext,
                               void *pCommandContext);



/** \brief Calbacks from the SDK for SBP Sink events. */
typedef struct
{
  /** Called when a ServiceResponse message is received from the service. */
  VNCCDBEndpointServiceResponseCallback *pResponseCallback;

  /** Called when a response with object data is recevied. */
  VNCSBPSinkGetResponseCallback *pGetResponseCallback;
  /** Called when a response to set object data is recevied. */
  VNCSBPSinkSetResponseCallback *pSetResponseCallback;
  /** Called when a response to subscribe to an object is received. */
  VNCSBPSinkSubscribeResponseCallback *pSubscribeResponseCallback;
  /** Called when a reponse to a cancel is recevied. */
  VNCSBPSinkCancelResponseCallback *pCancelResponseCallback;
  /** Called when a request fails or times out. */
  VNCSBPSinkRequestErrorCallback *pRequestErrorCallback;
  /** Called when an authentication challenge is received. */
  VNCSBPSinkServiceAuthChallengeCallback *pServiceAuthChallengeCallback;
  /** Called when a service responds to an aliveness check. */
  VNCSBPSinkServiceAliveCallback *pServiceAliveCallback;
} VNCSBPSinkCallbacks;


/** @} */



/** \defgroup sbp-source SBP source functions
 * @{
 */

/** \brief Called by the SDK when a Get request is received.
 *
 * The SDK calls this function when the sink has requested the state of an
 * object. The application should respond by passing the current state of the
 * object to the VNCSBPSourceSendGetResponse() function either in the context of
 * this callback or at a later point.
 *
 * This callback is invoked for either an explicit Get request, or for a
 * Subscribe request where the SBP Sink has requested periodic updates.
 * If necessary the application can differentiate between these two cases
 * based on the command parameter.
 *
 * If authentication is required, the application can call an appropriate
 * authentication function prior to sending the response. Currently the only
 * such function defined is VNCSBPSourceSendServiceAuthChallenge() for
 * service-specific authentication. In this case, the application must call
 * VNCSBPSourceSendGetResponse() in the same way once the authentication response
 * is received in VNCSBPSourceServiceAuthResponseCallback().
 *
 * If the application wishes to handle the request using
 * VNCSBPSourceSendGetResponse() then it should return ::VNCCDBErrorNone and
 * send a response as outlined above. If the application doesn't wish to handle
 * the Get (for example, if the object UID isn't recognised) then it can
 * populate the errorCode parameter with an appropriate SBP error and return
 * ::VNCCDBErrorLocalSBPError. This will cause the SDK to respond with the
 * specified error response immediately. Any error code other than
 * ::VNCCDBErrorLocalSBPError and ::VNCCDBErrorNone will result in a
 * "feature not supported" response being sent.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID from VNCCDBEndpointAddSBPSource().
 * \param[in]   pSourceContext  Application per-service context.
 * \param[in]   command         Command type (Get or Subscribe).
 * \param[in]   objectUID       Object for which state has been requested.
 * \param[in]   errorCode       SBP error to use if local SBP error is returned.
 *
 * \retval ::VNCCDBErrorNone            Get response has been/will be
 *                                      sent asynchronously with
 *                                      VNCSBPSourceSendGetResponse().
 * \retval ::VNCCDBErrorLocalSBPError   Send errorCode as immediate response
 *                                      with empty payload.
 */
typedef VNCCDBError VNCCALL
VNCSBPSourceGetRequestCallback(VNCCDBEndpoint *pEndpoint,
                               VNCCDBServiceId id,
                               void *pSourceContext,
                               VNCSBPCommandType command,
                               VNCSBPUID objectUID,
                               VNCSBPProtocolError *errorCode);


/** \brief Called by the SDK when a Set request is received.
 *
 * The SDK calls this function when the SBP Sink requests the state of an
 * object be updated with supplied data. The application should respond by
 * effecting the requested changes (or not) and then passing an appropriate
 * error code to VNCSBPSourceSendSetResponse(), or alternatively updating the
 * errorCode parameter and returning ::VNCCDBErrorLocalSBPError.
 *
 * If authentication is required, the application can call an appropriate
 * authentication function prior to sending the response. Currently the only
 * such function defined is VNCSBPSourceSendServiceAuthChallenge() for
 * service-specific authentication. In this case, the application must call
 * VNCSBPSourceSendSetResponse() in the same way once the authentication
 * response is received in VNCSBPSourceServiceAuthResponseCallback().
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID from VNCCDBEndpointAddSBPSource().
 * \param[in]   pSourceContext  Application per-service context.
 * \param[in]   objectUID       Object for which update has been requested.
 * \param[in]   payload         Pointer to serialized payload supplying updated
 *                              object state.
 * \param[in]   payloadSize     Bytes available in `payload`.
 * \param[in]   errorCode       SBP error to use if local SBP error is returned.
 *
 * \retval VNCCDBErrorNone              Set response has been / will be
 *                                      sent asynchronously with
 *                                      VNCSBPSourceSendSetResponse().
 * \retval VNCCDBErrorLocalSBPError     Send errorCode as immediate response.
 */
typedef VNCCDBError VNCCALL
VNCSBPSourceSetRequestCallback(VNCCDBEndpoint *pEndpoint,
                               VNCCDBServiceId id,
                               void *pSourceContext,
                               VNCSBPUID objectUID,
                               const vnc_uint8_t *payload,
                               size_t payloadSize,
                               VNCSBPProtocolError *errorCode);


/** \brief Called by the SDK when a Subscribe request is received.
 *
 * The SDK calls this function when the SBP Sink subscribes to an object.
 * The trigger parameter indicates the type of subscription requested, which
 * will be ::VNCSBPSubscriptionTypeRegularInterval for periodic updates,
 * ::VNCSBPSubscriptionTypeOnChange for updates when the state of an object
 * changes or ::VNCSBPSubscriptionTypeAutomatic to allow the Source to choose
 * the most appropriate update method.
 *
 * The application can reject the subscription request outright by setting
 * errorCode to the appropriate error code and returning
 * ::VNCCDBErrorLocalSBPError. In any other case if trigger's value is
 * ::VNCSBPSubscriptionTypeAutomatic, the application must update the value
 * of trigger to indicate to the SDK the type of subscription chosen. It is
 * invalid to change the value of this parameter if the sink requested a
 * specific subscription type.
 *
 * If periodic subscriptions are selected, the SDK will automatically call
 * VNCSBPSourceGetRequestCallback() at the correct interval to obtain
 * the update to send to the Sink. If desired, the application can change the
 * value of the intervalMs parameter to specify the interval at which these
 * requests should occur - applications should always do this if the Sink
 * specified automatic subscription types.
 *
 * In the case of "on change" notifications, it is up to the application to
 * send updates whenever the state of the object changes by calling
 * VNCSBPSourceSendGetResponse() passing in ::VNCSBPCommandTypeSubscribe as the
 * command parameter to that function.
 *
 * If authentication is required, the application can call an appropriate
 * authentication function prior to sending the response. Currently the only
 * such function defined is VNCSBPSourceSendServiceAuthChallenge() for
 * service-specific authentication. In this case, the application must call
 * VNCSBPSourceSendSetResponse() in the same way once the authentication
 * response is received in VNCSBPSourceServiceAuthResponseCallback().
 *
 * \note Even if authentication is required, the application must still update
 *       the trigger and/or intervalMs parameters as required before returning
 *       from this function.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID from VNCCDBEndpointAddSBPSource().
 * \param[in]   pSourceContext  Application per-service context.
 * \param[in]   objectUID       Object for which updates have been requested.
 * \param[in,out] trigger       Requested subscription type.
 * \param[in,out] intervalMs    Requested update interval in milliseconds.
 * \param[out]  errorCode       SBP error to use if local SBP error is returned.
 *
 * \retval ::VNCCDBErrorNone            Subscribe response has been / will be
 *                                      sent asynchronously with
 *                                      VNCSBPSourceSendSubscribeResponse().
 * \retval ::VNCCDBErrorLocalSBPError   Send errorCode as immediate response.
 */
typedef VNCCDBError VNCCALL
VNCSBPSourceSubscribeRequestCallback(VNCCDBEndpoint *pEndpoint,
                                     VNCCDBServiceId id,
                                     void *pSourceContext,
                                     VNCSBPUID objectUID,
                                     VNCSBPSubscriptionType *trigger,
                                     vnc_uint32_t *intervalMs,
                                     VNCSBPProtocolError *errorCode);


/** \brief Called by the SDK when a Cancel request is received.
 *
 * The SDK calls this function to notify the application that a prior request
 * has been cancelled by the Sink. Applications will typically only use this
 * to cancel any on-change subscriptions which are currently in effect. The SDK
 * will automatically cancel any interval-based subscriptions and perform any
 * other cleanup associated with an outstanding request regardless of whether
 * an application registers this callback, as well as sending the appropriate
 * responses to both the original command and the Cancel request itself.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID from VNCCDBEndpointAddSBPSource().
 * \param[in]   pSourceContext  Application per-service context.
 * \param[in]   commandType     Command type being cancelled.
 * \param[in]   objectUID       Object for which command is being cancelled.
 *
 * \return      Reserved for future expansion, always return ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCSBPSourceCancelRequestCallback(VNCCDBEndpoint *pEndpoint,
                                  VNCCDBServiceId id,
                                  void *pSourceContext,
                                  VNCSBPCommandType commandType,
                                  VNCSBPUID objectUID);


/** \brief Send a response to a Get request from the Sink.
 *
 * This function is called in response to VNCSBPSourceGetRequestCallback()
 * being invoked by the SDK, either in the context of that callback or at any
 * later point by the application. This function should only be used for a given
 * request if the callback returns ::VNCCDBErrorNone.
 *
 * To create the payload for the response, the application should call
 * VNCSBPSerializeBeginPacket(), then the various serialization functions to
 * add the object's parameters as required. Once all parameters have been
 * added, the application should call VNCSBPSerializeEnd() to close the packet,
 * and then VNCSBPSerializeFinalize() to obtain the payload and payloadSize
 * parameters to pass into this function.
 *
 * Alternatively, applications can return an empty payload and set errorCode to
 * an appropriate error value. The error value ::VNCSBPProtocolErrorContinue must
 * not be used as this is reserved for "continue" packets. Setting errorCode to
 * ::VNCSBPProtocolErrorContinue will cause this function to return
 * ::VNCCDBErrorInvalidParameter.
 *
 * Any issues sending the response to the Sink will typically be returned via
 * VNCSBPSourceResponseErrorCallback().
 *
 * The 5 second SBP timeout does not apply to an application waiting to send a
 * response as the SDK will automatically send "continue" packets as necessary
 * to keep the request active for as long as it takes the application to send
 * a response. The SBP Sink can, of course, cancel the pending request at
 * any point (in which case the application will receive a call to
 * VNCSBPSourceCancelRequestCallback()).
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID from VNCCDBEndpointAddSBPSource().
 * \param[in]   pCommandContext Application per-command context for
 *                              VNCSBPSourceResponseErrorCallback().
 * \param[in]   command         ::VNCSBPCommandTypeGet or ::VNCSBPCommandTypeSubscribe.
 * \param[in]   objectUID       Object for which request was received.
 * \param[in]   errorCode       ::VNCSBPProtocolErrorNone to return the object, or
 *                              an appropriate other error otherwise. This must
 *                              not be ::VNCSBPProtocolErrorContinue.
 * \param[in]   payload         Payload for response packet as from
 *                              VNCSBPSerializeFinalize().
 * \param[in]   payloadSize     Bytes in payload as from VNCSBPSerializeFinalize().
 *
 * \return      ::VNCCDBErrorNone on success, an appropriate error code otherwise.
 */
typedef VNCCDBError VNCCALL
VNCSBPSourceSendGetResponse(VNCCDBEndpoint *pEndpoint,
                            VNCCDBServiceId id,
                            void *pCommandContext,
                            VNCSBPCommandType command,
                            VNCSBPUID objectUID,
                            VNCSBPProtocolError errorCode,
                            const vnc_uint8_t *payload,
                            size_t payloadSize);


/** \brief Send a response to a Set request from the Sink.
 *
 * This function is called in response to VNCSBPSourceSetRequestCallback()
 * being invoked by the SDK, either in the context of that callback or at any
 * later point by the application. This function should only be used for a given
 * request if the callback returns ::VNCCDBErrorNone.
 *
 * The errorCode parameter should be ::VNCSBPProtocolErrorNone if the Set was
 * successful, or an appropriate error code otherwise. The error code
 * ::VNCSBPProtocolErrorContinue must not be used as this is reserved for
 * "continue" packets. Setting errorCode to ::VNCSBPProtocolErrorContinue will
 * cause this function to return ::VNCCDBErrorInvalidParameter.
 *
 * Any issues sending the response to the Sink will typically be returned via
 * VNCSBPSourceResponseErrorCallback().
 *
 * The 5 second SBP timeout does not apply to an application waiting to send a
 * response as the SDK will automatically send "continue" packets as necessary
 * to keep the request active for as long as it takes the application to send
 * a response. The SBP Sink can, of course, cancel the pending request at
 * any point (in which case the application will receive a call to
 * VNCSBPSourceCancelRequestCallback()).
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID from VNCCDBEndpointAddSBPSource().
 * \param[in]   pCommandContext Application per-command context for
 *                              VNCSBPSourceResponseErrorCallback().
 * \param[in]   objectUID       Object for which request was received.
 * \param[in]   errorCode       ::VNCSBPProtocolErrorNone to indicate success, or
 *                              an appropriate other error otherwise. This must
 *                              not be ::VNCSBPProtocolErrorContinue.
 *
 * \return      ::VNCCDBErrorNone on success, an appropriate error code otherwise.
 */
typedef VNCCDBError VNCCALL
VNCSBPSourceSendSetResponse(VNCCDBEndpoint *pEndpoint,
                            VNCCDBServiceId id,
                            void *pCommandContext,
                            VNCSBPUID objectUID,
                            VNCSBPProtocolError errorCode);


/** \brief Send a response to a Subscribe request from the Sink.
 *
 * This function is called in response to
 * VNCSBPSourceSubscribeRequestCallback() being invoked by the SDK, either in
 * the context of that callback or at any later point by the application. This
 * function should only be used for a given request if the callback returns
 * ::VNCCDBErrorNone.
 *
 * The errorCode parameter should be ::VNCSBPProtocolErrorNone if the request 
 * was successful, or an appropriate error code otherwise. The error code
 * ::VNCSBPProtocolErrorContinue must not be used as this is reserved for
 * "continue" packets. Setting errorCode to ::VNCSBPProtocolErrorContinue will
 * cause this function to return ::VNCCDBErrorInvalidParameter.
 *
 * Any issues sending the response to the Sink will typically be returned via
 * VNCSBPSourceResponseErrorCallback().
 *
 * The 5 second SBP timeout does not apply to an application waiting to send a
 * response as the SDK will automatically send "continue" packets as necessary
 * to keep the request active for as long as it takes the application to send
 * a response. The SBP Sink can, of course, cancel the pending request at
 * any point (in which case the application will receive a call to
 * VNCSBPSourceCancelRequestCallback()).
 *
 * \note        Even if an application choose to send an asynchronous response
 *              via this function, the trigger and intervalMs parameters to
 *              VNCSBPSourceSubscribeRequestCallback() still must be updated
 *              within that function if required. In the case of interval-based
 *              updates, however, the SDK won't begin to query the application
 *              until a successful response is sent by either method.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID from VNCCDBEndpointAddSBPSource().
 * \param[in]   pCommandContext Application per-command context for
 *                              VNCSBPSourceResponseErrorCallback().
 * \param[in]   objectUID       Object for which request was received.
 * \param[in]   errorCode       ::VNCSBPProtocolErrorNone to indicate success, or
 *                              an appropriate other error otherwise. This must
 *                              not be ::VNCSBPProtocolErrorContinue.
 *
 * \return      ::VNCCDBErrorNone on success, an appropriate error code otherwise.
 */
typedef VNCCDBError VNCCALL
VNCSBPSourceSendSubscribeResponse(VNCCDBEndpoint *pEndpoint,
                                  VNCCDBServiceId id,
                                  void *pCommandContext,
                                  VNCSBPUID objectUID,
                                  VNCSBPProtocolError errorCode);


/** \brief Send a service-specific authentication challenge to the Sink.
 *
 * This function can be called to send a service-specific authentication request
 * to the Sink in response to any of the requests. The objectUID and commandType
 * parameters must be set to the same values as the incoming request, and the
 * definition of the payload content depends entirely on the service in
 * question.
 *
 * When a response to the authentication challenge is received from the Sink,
 * VNCSBPSourceServiceAuthResponseCallback() will be invoked by the SDK.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID from VNCCDBEndpointAddSBPSource().
 * \param[in]   pCommandContext Application per-command context for
 *                              future callbacks.
 * \param[in]   commandType     Request type for which authentication is
 *                              required.
 * \param[in]   objectUID       Object for which request was received.
 * \param[in]   payload         Payload of service-specific challenge.
 * \param[in]   payloadSize     Number of valid bytes in payload.
 *
 * \return      ::VNCCDBErrorNone on success, an appropriate error code otherwise.
 */
typedef VNCCDBError VNCCALL
VNCSBPSourceSendServiceAuthChallenge(VNCCDBEndpoint *pEndpoint,
                                     VNCCDBServiceId id,
                                     void *pCommandContext,
                                     VNCSBPCommandType commandType,
                                     VNCSBPUID objectUID,
                                     const vnc_uint8_t *payload,
                                     size_t payloadSize);


/** \brief Called by the SDK with a response to a service-specific auth challenge.
 *
 * This function is called by the SDK when a response is received from the Sink
 * to a challenge initiated by VNCSBPSourceSendServiceAuthChallenge(). The
 * application can then choose how to respond to the original request based on
 * this response.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID from VNCCDBEndpointAddSBPSource().
 * \param[in]   pSourceContext  Application per-service context.
 * \param[in]   pCommandContext Application per-command context, as passed to
 *                              VNCSBPSourceSendServiceAuthChallenge().
 * \param[in]   commandType     Original request type.
 * \param[in]   objectUID       Object for which request was received.
 * \param[in]   errorCode       Error code copied directly from response packet.
 * \param[in]   payload         Payload of service-specific response
 * \param[in]   payloadSize     Number of valid bytes in payload.
 *
 * \return      Reserved for future expansion, always return ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCSBPSourceServiceAuthResponseCallback(VNCCDBEndpoint *pEndpoint,
                                        VNCCDBServiceId id,
                                        void *pSourceContext,
                                        void *pCommandContext,
                                        VNCSBPCommandType commandType,
                                        VNCSBPUID objectUID,
                                        VNCSBPProtocolError errorCode,
                                        const vnc_uint8_t *payload,
                                        size_t payloadSize);


/** \brief Called by the SDK on a failure to send a response.
 *
 * This function is invoked by the SDK to indicate to the application that an
 * error occurred sending a response back to the Sink. The sdkError parameter
 * indicates the basic type of the error and, if this has either of the
 * values ::VNCCDBErrorLocalSBPError or ::VNCCDBErrorRemoteSBPError then the
 * sbpError field is also populated with the SBP protocol error code.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID from VNCCDBEndpointAddSBPSource().
 * \param[in]   pSourceContext  Application per-service context.
 * \param[in]   pCommandContext Application per-command context, as passed to
 *                              the response-sending command.
 * \param[in]   commandType     Original request type.
 * \param[in]   objectUID       Object for which request was received.
 * \param[in]   sdkError        Basic type of error.
 * \param[in]   sbpError        For SBP protocol errors, sdkError will be
 *                              ::VNCCDBErrorLocalSBPError or
 *                              ::VNCCDBErrorRemoteSBPError, and this field
 *                              will be populated as appropriate.
 *
 * \return      Reserved for future expansion, always return ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCSBPSourceResponseErrorCallback(VNCCDBEndpoint *pEndpoint,
                                  VNCCDBServiceId id,
                                  void *pSourceContext,
                                  void *pCommandContext,
                                  VNCSBPCommandType commandType,
                                  VNCSBPUID objectUID,
                                  VNCCDBError sdkError,
                                  VNCSBPProtocolError sbpError);


/** \brief Called by the SDK when it needs to forcibly shut down a service.
 *
 * This function is invoked by the SDK to indicate that the specified service
 * is being forcibly reset, typically due to receiving a corrupt SBP message
 * or other irrecoverable error. The SDK handles sending an appropriate "reset"
 * response to the Sink, so this callback acts solely as a notification to
 * the application to free any resources associated with the service in the same
 * way as if the "stop" callback was invoked. Unlike the "stop" callback,
 * however, the application should not send any CDB or SBP messages in response.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   id              Service ID from VNCCDBEndpointAddSBPSource().
 * \param[in]   pSourceContext  Application per-service context.
 *
 * \return      Reserved for future expansion, always return ::VNCCDBErrorNone.
 */
typedef VNCCDBError VNCCALL
VNCSBPSourceResetServiceCallback(VNCCDBEndpoint *pEndpoint,
                                 VNCCDBServiceId id,
                                 void *pSourceContext);


/** \brief Callbacks from the SDK for SBP Source events. */
typedef struct
{
  /** Called when a StartService message is received for this service. */
  VNCCDBEndpointStartServiceCallback *pStartCallback;
  /** Called when a StopService message is received for this service. */
  VNCCDBEndpointStopServiceCallback *pStopCallback;
  /** Called when a ServiceResponse message is received from the service. */
  VNCCDBEndpointServiceResponseCallback *pResponseCallback;

  /** Called when a service needs to forcibly shut down. **/
  VNCSBPSourceResetServiceCallback *pResetCallback;
  /** Called when a get request is recevied. **/
  VNCSBPSourceGetRequestCallback *pGetRequestCallback;
  /** Called when a set request is received. **/
  VNCSBPSourceSetRequestCallback *pSetRequestCallback;
  /** Called when a subscribe request is received. **/
  VNCSBPSourceSubscribeRequestCallback *pSubscribeRequestCallback;
  /** Called when a cancel request is received. **/
  VNCSBPSourceCancelRequestCallback *pCancelRequestCallback;
  /** Called when an authentication response is received. **/
  VNCSBPSourceServiceAuthResponseCallback *pServiceAuthResponseCallback;
  /** Called when on failure to send a response. **/
  VNCSBPSourceResponseErrorCallback *pResponseErrorCallback;
} VNCSBPSourceCallbacks;


/** @} */


/** \brief Attach an SBP Sink to a remote service.
 *
 * This function should be called to attach to a remote service and communicate
 * with it via SBP.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   pContext        An application context, passed into each SBP
 *                              Sink callback function in pSinkCallbacks.
 * \param[in]   serviceId       The ID of the remote service as obtained from
 *                              VNCCDBEndpointServicesSupportedCallback().
 * \param[in]   pSinkCallbacks  Callback functions for SBP operation.
 * \param[in]   sinkCallbacksSize Size of structure pointed to by pSinkCallbacks.
 *
 * \retval ::VNCCDBErrorNone The Sink was created successfully.
 * \retval ::VNCCDBErrorInvalidParameter One or more of the parameters is
 * invalid.
 * \retval ::VNCCDBErrorOutOfMemory A dynamic memory allocation failed.
 */
typedef VNCCDBError VNCCALL
VNCCDBEndpointAddSBPSink(VNCCDBEndpoint *pEndpoint,
                         void *pContext,
                         VNCCDBServiceId serviceId,
                         const VNCSBPSinkCallbacks *pSinkCallbacks,
                         size_t sinkCallbacksSize);



/** \brief Attach an SBP Source to a CDB Endpoint.
 *
 * This function should be called to register a local service to be provided
 * by the specified CDB endpoint. This function must be called prior to calling
 * VNCCDBEndpointStart().
 *
 * The service parameter should point to a structure which the caller has
 * initialised with the appropriate name, version, configuration and
 * access control values. Upon a successful return from this function, the
 * serviceId field will have been initialised with the ID that the endpoint
 * has assigned to the service - this ID should be used in all future calls
 * into the SBP functions.
 *
 * \param[in]   pEndpoint       Endpoint from VNCCDBEndpointCreate().
 * \param[in]   pContext        An application context, passed into each SBP
 *                              Source callback function in pSourceCallbacks.
 * \param[in,out] service       A populated VNCCDBService structure with all
 *                              fields except serviceId initialized to the
 *                              appropriate values for the service.
 * \param[in]   pSourceCallbacks Callback functions for SBP operation.
 * \param[in]   sourceCallbacksSize Size of structure pointed to by
 *                                  pSourceCallbacks.
 *
 * \retval ::VNCCDBErrorNone The Source was created successfully.
 * \retval ::VNCCDBErrorInvalidParameter One or more of the parameters is
 * invalid.
 * \retval ::VNCCDBErrorOutOfMemory A dynamic memory allocation failed.
 */
typedef VNCCDBError VNCCALL
VNCCDBEndpointAddSBPSource(VNCCDBEndpoint *pEndpoint,
                           void *pContext,
                           VNCCDBService *service,
                           size_t serviceSize,
                           const VNCSBPSourceCallbacks *pSourceCallbacks,
                           size_t sourceCallbacksSize);


#ifdef __cplusplus
}
#endif

#endif /* ifndef UUID_269fdfdc_51e3_483e_80cb_573aa4c31d7d */
